﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web.UI;

namespace Framework.Web.UI
{
    /// <summary>
    /// ファイルダウンロード用ページの基底クラス
    /// 
    /// 通常の画面内でダウンロードを行うと、ダウンロード後に画面への応答がなくなり
    /// 画面更新が行われないため本クラスを継承した専用のダウンロードページを用意する。
    /// URLを「/DownloadPage.aspx」にしておくと、変更なしで使用できます。
    /// </summary>
    public class DownloadPage : System.Web.UI.Page
    {
        /// <summary>
        /// 本クラスを継承したダウンロードページのURL
        /// </summary>
        public static string PageUrl = "/DownloadPage.aspx";

        public static string SessionGroup = "Framework.Web.DownloadPage";

        /// <summary>
        /// 指定されたファイルをダウンロードします。
        /// </summary>
        public static void Download(Page page, string physicalPath, string downloadName, bool deleteFile)
        {
            SessionMgr.Remove(SessionGroup, false);

            //セッションに保存
            SessionMgr.Save(SessionGroup, "PhysicalPath", physicalPath, false);
            SessionMgr.Save(SessionGroup, "DownloadName", downloadName, false);
            SessionMgr.Save(SessionGroup, "DeleteFile", deleteFile, false);

            //ダウンロード用のページを別ウィンドウで開く
            JavaScript.RegisterWindowOpen(page, PageUrl, "DownloadPage");
        }

        /// <summary>
        /// 指定されたデータをダウンロードします。
        /// </summary>
        public static void Download(Page page, byte[] data, string downloadName)
        {
            SessionMgr.Remove(SessionGroup, false);

            //セッションに保存
            SessionMgr.Save(SessionGroup, "Data", data, false);
            SessionMgr.Save(SessionGroup, "DownloadName", downloadName, false);

            //ダウンロード用のページを別ウィンドウで開く
            JavaScript.RegisterWindowOpen(page, PageUrl, "DownloadPage");
        }

        /// <summary>
        /// OnPreRender
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            var data = SessionMgr.Load<byte[]>(SessionGroup, "Data", false, null);
            var path = SessionMgr.Load<string>(SessionGroup, "PhysicalPath", false, null);
            var name = SessionMgr.Load<string>(SessionGroup, "DownloadName", false, null);
            var delete = SessionMgr.Load<bool>(SessionGroup, "DeleteFile", false, ()=>false);

            name = Server.UrlEncode(name);

            try
            {
                //ダウンロード
                this.Response.ClearContent();
                this.Response.AddHeader("Content-Disposition", "attachment;filename={0}".Fmt(name));
                this.Response.ContentType = GetContentType(name);
                if (path != null)
                {
                    this.Response.WriteFile(path);
                }
                else if (data != null)
                {
                    this.Response.BinaryWrite(data);
                }
                this.Response.Flush();
                this.Response.End();
            }
            finally
            {
                SessionMgr.Remove(SessionGroup, false);

                if (delete == true)
                {
                    System.IO.File.Delete(path);
                }
            }
        }

        /// <summary>
        /// ファイル拡張子からContentTypeを推定します。わからない場合はnullを返します。
        /// </summary>
        /// <param name="filename"></param>
        /// <returns></returns>
        public static string GetContentType(string filename)
        {
            var ext = System.IO.Path.GetExtension(filename);
            if (ext.IsBlank() == true) return null;

            ext = ext.TrimStart('.').ToLower();
            if (ext == "xls" || ext == "xlsx") return "application/msexcel";
            if (ext == "doc" || ext == "docx") return "application/msword";
            if (ext == "ppt" || ext == "pptx") return "application/mspowerpoint";
            if (ext == "pdf") return "application/pdf";
            if (ext == "zip") return "application/zip";
            if (ext == "csv") return "text/comma-separated-values";
            if (ext == "tsv") return "text/tab-separated-values";
            if (ext == "txt") return "text/plain";
            if (ext == "xml") return "text/xml";

            return null;
        }
    }
}
